﻿
/**
* This class is responsible for maintaining a cart and the addition and removal of
* products in the cart.
*
* @author Macromedia
* @version 1.0
*/

import mx.controls.*;

class Cart {
	var cart_dg:DataGrid;
	var clearCart_button:Button;
	var m_cart_so:SharedObject;
	var m_parent_mc:MovieClip;
	// constructor
	function Cart(parent_mc:MovieClip) {
		this.m_cart_so = SharedObject.getLocal("cart");
		if (this.m_cart_so.data.products == undefined) {
			this.m_cart_so.data.products = new Array();
			this.m_cart_so.flush();
		}
		m_parent_mc = parent_mc.createEmptyMovieClip("cart_mc", 10);
	}
	//
	/*
		Method:	clearCart()
		Purpose: this method removes all items from the cart SharedObject and refreshes the cart Data Grid on the Stage.
		Parameters: none
		Returns: Void
	*/
	public function clearCart():Void {
		this.m_cart_so.data.products = new Array();
		this.m_cart_so.flush();
		refreshCart();
	}
	//
	/*
		Name: getCart()
		Purpose: returns the SharedObject Array of cart items.
		Parameters: none
		Returns: Array
	*/
	private function getCart():Array {
		return this.m_cart_so.data.products;
	}
	//
	/*
		Name: addProduct()
		Purpose: adds a product to the SharedObject Array of cart items.
		Parameters: quantity (Number)
					productObj (Object)
		Returns: Void
	*/
	public function addProduct(quantity:Number, productObj:Object):Void {
		var productPos:Number = checkProductInCart(productObj.id);
		if (productPos == -1) {
			this.m_cart_so.data.products.push({quantity:quantity, productID:productObj.id, name:productObj.name, price:productObj.price});
		} else {
			this.m_cart_so.data.products[productPos].quantity += quantity;
		}
		this.m_cart_so.flush();
		refreshCart();
	}
	//
	/*
		Name: checkProductInCart()
		Purpose: checks to see if a particular productID exists in the cart. If the specified productID wasn't found in the cart, this function returns -1.
		Parameters: productID (Number)
		Returns: Number
	*/
	private function checkProductInCart(productID:Number):Number {
		var products_array:Array = this.m_cart_so.data.products;
		for (var i = 0; i<products_array.length; i++) {
			if (products_array[i].productID == productID) {
				return i;
			}
		}
		return -1;
	}
	//
	/*
		Name: init()
		Purpose: This method initializes the application and adds the components to the Stage
		Parameters: none
		Returns: Void
	*/
	public function init():Void {
		// assign a shortcut to the Cart class for use within nested functions.
		var thisCart:Cart = this;
		// dynamically add a DataGrid to the Stage and give it an instance name of "cart_dg".
		var cart:DataGrid = this.m_parent_mc.createClassObject(mx.controls.DataGrid, "cart_dg", 998);
		cart.columnNames = ["quantity", "name", "price", "subtotal"];
		cart.setSize(400, 120);
		var quantity_dgc = cart.getColumnAt(0);
		quantity_dgc.width = 60;
		quantity_dgc.headerText = "Quantity";
		var name_dgc = cart.getColumnAt(1);
		name_dgc.width = 180;
		name_dgc.headerText = "Product";
		name_dgc.labelFunction = function(item:Object) {
			if (item != undefined) {
				return item.name;
			}
		};
		var price_dgc = cart.getColumnAt(2);
		price_dgc.width = 70;
		price_dgc.headerText = "Price";
		price_dgc.labelFunction = function(item:Object) {
			if (item != undefined) {
				return "$"+item.price;
			}
		};
		var subtotal_dgc = cart.getColumnAt(3);
		subtotal_dgc.width = 70;
		subtotal_dgc.headerText = "Subtotal";
		subtotal_dgc.labelFunction = function(item:Object) {
			if (item != undefined) {
				return "$"+(item.price*item.quantity);
			}
		};
		cart.move(77, 280);
		//
		var deleteProductListener:Object = new Object();
		deleteProductListener.change = function(evt:Object) {
			var myClickHandler = function (alertEvt:Object) {
				if (alertEvt.detail == Alert.OK) {
					thisCart.removeFromCart(evt.target.selectedItem.productID);
				}
			};
			Alert.show("Are you sure you want to delete this item?", "Remove Item From Cart", Alert.OK | Alert.CANCEL, thisCart.m_parent_mc, myClickHandler, "stockIcon", Alert.OK);
		};
		cart.addEventListener("change", deleteProductListener);
		//
		var clearCart = this.m_parent_mc.createClassObject(mx.controls.Button, "clearCart_button", 990, {label:"Clear Cart"});
		clearCart.move(560, 345);
		clearCart.clickHandler = function() {
			var myClickHandler = function (evt) {
				if (evt.detail == Alert.OK) {
					thisCart.clearCart();
				}
			};
			Alert.show("Are you sure you want to clear your cart?", "Clear Cart", Alert.OK | Alert.CANCEL, _root, myClickHandler, "stockIcon", Alert.OK);
		};
		thisCart.refreshCart();
	}
	//
	/*
			Name: refreshCart()
			Purpose: This method refreshes the cart DataGrid instance on the Stage to synch the contents of the Grid with the contents of the SharedObject cart.
			Parameters: none
			Returns: Void
		*/
	private function refreshCart():Void {
		var product_array:Array = getCart();
		// if there are no items in the array, disable the DataGrid.
		m_parent_mc.cart_dg.enabled = (product_array.length != 0);
		m_parent_mc.cart_dg.dataProvider = product_array;
	}
	//
	/*
		Name: removeFromCart()
		Purpose: This method removes a product from the cart based on the supplied productID. If the specified productID value wasn't found in the cart this method does nothing.
		Parameters: productID (Number)
		Returns: Void
	*/
	public function removeFromCart(productID:Number):Void {
		var index:Number = checkProductInCart(productID);
		// if the item was found in the cart, remove it using the Array.splice method.
		if (index != -1) {
			this.m_cart_so.data.products.splice(index, 1);
		}
		refreshCart();
	}
}
